// ******************************************************
// Example program using Microsoft C v7.0 and
// the Capital Equipment Corp CEC488 GPIB interface card.
// To use another interface card, modify the GPIB subroutines
// where indicated.

// The SR780 is assumed to be at address 10 (default).

// Link this object file with ieee488.lib (or the library for your GPIB card).

// ******************************************************
// Connect the Source output to the Ch1 A and Ch2 A inputs!!
// ******************************************************

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <math.h>
#include <ieee-c.h>   //This is the CEC header file
			    //Use the .h file for your interface card.
                      //send(), enter(), transmit(), rarray(), tarray() are
			    //CEC routines defined in ieee-c.h

char recv[80];        //global GPIB receive string
char cmd[80];         //global GPIB command string
float rxBuff[401];    //array of IEEE floats to receive binary data 
float txBuff[2048];   //array of IEEE floats to send binary data
int status, length;   //CEC interface routines use these variables

#define sr780 10      // default GPIB address for SR780

// subroutines at the end of this listing
// you may need to modify these for your interface card!
void TxGpib (int, char *);
void GetGpib (int);
void TxSr780 (char *);
void GetSr780 (char *);
void WaitAvg (void);
double GetData (int, int);
void WaitIFC (void);
void GetSpace (void);

void main (void);

/* ***************************************************** */
void main (void)
     {
     // ****************************************************
     // You can see the commands received and responses generated by the SR780
     // by using [System]<Remote><View Qs> to display the interface buffers
     // on the screen.
     // ****************************************************

     int i, mode, type, number, nlen;
     long ack;
     double dispAVal, dispBVal, octreal, octimag;

     printf ("\nSR780 Example Program\n\n");

     // ****** initialize your GPIB card here ******
     initialize (21,0);	//controller
     settimeout (7500);	//GPIB timeout to 7.5 seconds


     // ****** Set the SR780 output interface to GPIB!! ******
     TxSr780 ("*CLS");  //clear all status words
     TxSr780 ("OUTX0"); //direct SR780 responses to GPIB interface

     // ****** set the Standard Event enable register to catch EXE and CME
     // Command errors will set the ESB bit in the Serial Poll status word
     TxSr780 ("*ESE 48"); //set bits 4 and 5

     // ****** Check communication by querying SR780 IDN string
     GetSr780 ("*IDN?");
     printf ("[GPIB IDN]  = %s\n\n", recv);

     printf ("Reset SR780...");
     TxSr780 ("*RST");       // reset the SR780 to a default state
     printf ("done\n\n");

/* ***************************************************** */
/* ****** set up FFT averaging for both displays ******* */
     TxSr780 ("FAVM 2,1");   //mode = rms
     TxSr780 ("FAVT 2,0");   //type = linear
     TxSr780 ("FAVN 2,250"); //number = 250
     TxSr780 ("FAVG 2,1");   //avg = On

     // ****** query these parameters for DisplayA ******
     GetSr780 ("FAVM? 0"); mode = atoi(recv);
     GetSr780 ("FAVT? 0"); type = atoi(recv);
     GetSr780 ("FAVN? 0"); number = atoi(recv);
     printf ("Avg mode = %d : type = %d : number = %d\n\n",mode,type,number);

     printf ("Avg start...");
     WaitAvg ();			   //start a linear average and wait until done
     printf ("done\n");

     TxSr780 ("ASCL 0; ASCL 1"); //autoscale the displays

/* ***************************************************** */
/* ****** read data points ****** */
     dispAVal = GetData (0,40);  //read DisplayA bin 40 (peak)
     dispBVal = GetData (1,40);  //read DisplayB bin 40
     printf ("DisplayA = %lf dBVpk\n",dispAVal);
     printf ("DisplayB = %lf dBVpk\n",dispBVal);

/* ***************************************************** */
/* ****** Binary transfer all of DisplayA spectrum ****** */
     //We need to send the DSPB?0 command WITHOUT waiting for
     //IFC in serial poll status since IFC will not be set until
     //AFTER the transfer is complete!
     //This section needs to be modified for your GPIB interface

     printf ("\nReading entire DisplayA...");
     TxGpib (sr780,"DSPB? 0");           //use TxGpib (don't wait for IFC)
     transmit ("MLA TALK 10", &status);  //make the PC listen, SR780 talk
     rarray (rxBuff, 1604, &length, &status); //binary read 1604 bytes
						          //401 points x 4 bytes/point
     WaitIFC ();     //serial poll until IFC set, ok to continue

     printf ("%d bytes read\n",length);       //actual number of bytes read

     // ****** print a few points from the spectrum
     for (i=37; i<44; i++) {printf ("bin %d = %f dBVpk\n", i, rxBuff[i]);}
     printf ("\n");


/* ***************************************************** */
/* ****** Turn on limit testing ****** */
     TxSr780 ("LCLR 0");   //first clear any existing limits for DisplayA
     TxSr780 ("LMAX 0,2"); //initialize 2 segments
     TxSr780 ("LSEG 0,0,0,12.032E3,-90.0,102.4E3,-90.0"); //upper segment
     TxSr780 ("LSEG 0,1,0,1.28E3,-90.0,8.96E3,-90.0");    //upper segment
     TxSr780 ("LTST 0,1"); //turn on testing

     // ****** take a new measurement
     printf ("Avg start...");
     WaitAvg ();
     printf ("done\n");

     GetSr780 ("LFAL?0");    //read limit test result
     printf ("Limit test = ");
     if (atoi(recv)==0) printf ("pass"); else printf ("fail");
     printf ("\n\n");

     // ****** clean up
     TxSr780 ("LCLR 0");   //clear limits, turn off testing
     TxSr780 ("FAVG 2,0"); //turn avg off

/* ***************************************************** */
/* ****** Load the arbitrary waveform buffer ****** */
     // construct a normalized ramp of 2048 points
     // each point is a 4 byte IEEE float between -1.0 and +1.0
     for (i=0; i<2048; i++) { txBuff[i] = ((float) i)/2048.F; }

     //This section needs to be modified for your GPIB interface

     printf ("Loading Arb buffer...");
     TxGpib (sr780,"ALOD? 2048");    //use TxGpib so we don't wait for IFC
     //SR780 will return a binary 1 to acknowledge
     transmit("mla talk 10",&status);    //make SR780 a talker, pc a listener
     rarray (&ack, 4, &length, &status); //binary read 4 bytes (long int)
     //return value should be 1
     printf ("%d bytes recvd [val=%ld]...",length,ack);

     transmit("mta listen 10",&status);  //make SR780 a listener, pc a talker
     tarray(txBuff,8192,1,&status);   //binary transfer of 8192 bytes to SR780
                                      //(2048 points x 4 bytes/point)
                                      //set EOI with last byte
     WaitIFC ();     //serial poll until IFC set, ok to continue
     printf ("done\n");

     TxSr780 ("STYP 3");             //change source type to Arb
     TxSr780 ("I1RG 4; I2RG 4");     //increase the input ranges
     TxSr780 ("FSPN 2,3.2E3");       //narrow the span to display the spectrum
     GetSpace ();                    //pause to look


/* ***************************************************** */
/* ****** Switch to Octave Analysis ****** */
     TxSr780 ("STYP 2");      //switch to Noise source
     TxSr780 ("NTYP 2");      //choose Pink noise
     TxSr780 ("MGRP 2,1");    //switch to Octave measurement group

     printf ("\n\nWait for octave settle...");
     GetSr780 ("DSPS?");    //clear any sticky bits in the Display status word
                            //and then wait until the settle bits become set
     do { GetSr780 ("DSPS?"); } while ( (atoi(recv)&0x0404) != 0x0404 );
     printf ("done\n\n");

     TxSr780 ("SVTR 0,5");    //save DisplayA to Trace5
     TxSr780 ("RCTR 1,5");    //recall Trace5 to DisplayB
     GetSr780 ("DSPN? 1"); nlen = atoi(recv); //query the length of DisplayB
     printf ("DisplayB length = %d\n",nlen);

/* ***************************************************** */
/* ****** Download an octave trace ****** */
     // Make Trace5 a constant -27 dBVpk for all bins.
     // Convert -27 dBVpk to Volts (10^(-27/20))=0.04467
     // Trace5 is already defined as an Octave measurement
     // so the downloaded points are assumed to be Volts squared!
     octreal = 0.04467*0.04467;    //(44.67 mV)^2 real part
     octimag = 0.0;                //no imaginary part

     //This section needs to be modified for your GPIB interface

     printf ("Loading Trace 5...");
     TxGpib (sr780,"TASC? 5,34");        //use TxGpib (no wait for IFC RDY)
                                         //TASC downloads ASCII data,
                                         //use TLOD to download binary data.
     transmit("mla talk 10",&status);    //make SR780 a talker, pc a listener
     rarray (&ack, 4, &length, &status); //binary read 4 bytes (long int)
                                         //return value should be 1
     printf ("%d bytes recvd [val=%ld]...",length,ack);

     // TASC (TLOD) replaces the data in an existing Trace (5).
     // The Trace measurement and length are not changed!
     // Download 34 bins of data (real, imaginary) with commas to separate,
     // and EOI at the end.
     // First 33 bins are the spectrum, last bin is the Total Power bin.
     // All points are complex.
     // For an octave measurement, the imaginary parts are all zero.
     transmit("mta listen 10",&status); //make SR780 a listener, pc a talker
     for (i=0; i<33; i++) {		    //download 33 octave bins
          sprintf (cmd,"DATA '%lf, '", octreal);  //real part
          transmit (cmd,&status);
          sprintf (cmd,"DATA '%lf, '", octimag);  //imaginary part
          transmit (cmd,&status);
          }
     // last bin is the total power bin
     sprintf (cmd,"DATA '%lf, '", octreal*33); //real part of total power bin
     transmit (cmd,&status);
     sprintf (cmd,"DATA '%lf' END", 0.0);    //imag part of total power bin,
     transmit (cmd,&status);                 //last value, terminate with EOI!
     WaitIFC ();     //serial poll until IFC set, ok to continue

     printf ("done\n");

     TxSr780 ("RCTR 1,5"); //recall Trace 5 to DisplayB to view the new data

/* ***************************************************** */
/* ****** Define a User Function ****** */
     // define Func1 = Oct(1)/Trace5
     TxSr780 ("USRO 1,14,-5,45"); //14=Oct(1), -5=divide, 45=Trace5
     TxSr780 ("MEAS 0,25");       //DisplayA measures UserFunc1
     TxSr780 ("YMAX 0,40");       //Change the top graph scale.
                                  //Func1 is Oct(1) normalized by Trace5.
                                  //The amplitudes are approximately 0 dB.
                                  //The Total Power bin is calculated from the
                                  //sum of the octave bins in the function,
					    //(it is not normalized to Trace5).
     // ****** end of main program
     }

/* ********************************************************************* */
void TxGpib (int address,char *command)
     // routine to transmit command to a GPIB address 
     // modify this routine for your GPIB interface

     {
     send (address,command,&status); // send() is the CEC routine to send
                                     // a string to an address.
                                     // Sets status = 0 to indicate result OK
     if (status != 0) {
          // handle transmit errors here
          printf ("\nCommand = %s\n",command);
          printf ("Error at device %d : status = %d\n",address,status);
          GetSpace ();
          }
     }
/* ********************************************************************* */
void GetGpib (int address)
     // routine to get an answer from a GPIB address 
     // modify this routine for your GPIB interface

     {
     char temp[80];
     
     enter (temp,80,&length,address,&status); // enter () is the CEC routine
                                     // to enter a string from an address.
                                     // Sets status = 0 to indicate result OK.
                                     // 80 is maxlen, actual received length
                                     // is stored in &length.
     if (status != 0) {
          // handle receive errors here
          printf ("\nError at device %d : status = %d\n",address,status);
          GetSpace ();
          }
     
     strcpy (recv, temp); //set global recv string
     }
/* ********************************************************************* */
void TxSr780 (char *command)
     // Send command to the SR780 and wait until IFC (bit7) is set in
     // the Serial Poll status (indicating that the command is finished).

     {
     TxGpib (sr780,command); // send command to sr780 address
     WaitIFC ();             //serial poll until IFC set, ok to continue
     }

/* ******************************************************************** */
void WaitIFC (void)
     {
     //serial poll until IFC (bit7) set (command done)
     //modify for your GPIB interface

     //stb is serial poll byte
     char stb;

     do {spoll(sr780,&stb,&status);} while (!(stb&128)); //IFC

     if (stb&32) {
          // If ESB bit set,
          // there must be a command error in the Standard Event status word.
          // Handle command errors here.
          TxGpib (sr780, "*ESR?"); //clear the Standard Event status word
          GetGpib (sr780);
          printf ("\nEXE error\n");
          GetSpace ();
          }
     }
/* ******************************************************************** */
void GetSr780 (char *getcmd)
     // query the sr780 for an answer
     // getcmd is the query command string

     {
     TxSr780 (getcmd); //send query command
     GetGpib (sr780);  //get response into global recv string
     }
/* ******************************************************************** */
void WaitAvg (void)
     {
     // routine to start a linear avg and wait until both displays are done

     int result, avgdone;

     avgdone=0;          //init avgdone status
     GetSr780 ("DSPS?"); //clear sticky bits in Display status word first

     TxSr780 ("STRT");   //start lin average measurement

     do { 
          GetSr780 ("DSPS?"); result=atoi(recv); //read display status word
          avgdone = avgdone | (result & 0x0202);
          //Check for AVGA and AVGB status bits until both have occurred.
          //Remember, they may not be set together in the same query!
          //Do not check for both in the same query!
          }
     while ((avgdone&0x0202)!=0x0202); // both AVGA and AVGB have occurred
     }
/* ******************************************************************** */
double GetData (int disp, int bin)
     {
     //routine to move the display marker to a bin and return the data value

     //move the marker in display disp to bin
     sprintf (cmd,"MBIN %d,%d",disp,bin);
     TxSr780 (cmd);

     sprintf (cmd,"DSPY? %d,%d",disp,bin); //read the data value at bin
     GetSr780 (cmd);
     return (atof(recv)); //return the value as a double
     }
/* ******************************************************************** */
void GetSpace (void)        /* Wait for space key */
     {
     char ch;

     while ( kbhit() ) getch();  // clear the keyboard buffer
     printf ("\n<Space> to continue, <Q> to quit ");
     do {
          ch = (char) getch ();
          if ((ch == 'q')||(ch =='Q')) exit(0); // exit
          }
     while  (ch!=' '); // continue
     printf ("\n");
     }
/* ******************************************************************* */


